Option Explicit

Const scriptName = "CtrlrMaker INI Generator"
Const scriptVer = "1.0.1"
Const releaseVer = "0.1.0"

'CtrlrMaker is an engine that helps generate MAME ctrlr directories.
'Copyright (C) 2003 Michael Miller

'Redistribution and use in source and binary forms, with or without
'modification, are permitted provided that the following conditions are met:

'Redistributions of source code must retain the above copyright notice,
'this list of conditions and the following disclaimer.
'Redistributions in binary form must reproduce the above copyright notice,
'this list of conditions and the following disclaimer in the documentation
'and/or other materials provided with the distribution.
'The name of the author may not be used to endorse or promote products
'derived from this software without specific prior written permission.

'THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
'IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
'OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
'IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
'INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
'NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
'DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
'THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
'(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
'THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.



Const errNoCtrlr = "Please specify a controller (-ctrlr)."
Const errNoSystem = "Please specify a system (-system)."
Const errArgumentsRequired = "Please specify a controller and system (-ctrlr and -system)."
Const errIncorrectArgumentCount = "Not enough arguments (they should come in pairs)."

Const msgComplete = "Done!"

' Search for ~Foo "bar"
Const kVariablePattern = "^(~\w+)\s*""(\w+)""\s*(?:#.*)?$"
Const kSectionPattern = "^\s*\[(\w+)\]\s*(?:#.*)?$"
Const kCommentPattern = "^\s*#.*$"
Const kBlankPattern = "^\s*$"
Const kNamePattern = "^\s*(\w+)\s*(?:#.*)?$"
Const kCommentChar = "#"

Const kActOnExtension = ".insrc"
Const kWriteToExtension = ".ini"
Const kDefinitionExtension = ".def"

Const kSubDirChar = "\"
Const kDefaultDestDir = "."
Const kDefaultSourceDir = ".\Sources"
Const kDefaultTempDir = ".\temp"
Const kDefaultDeleteTempDir = True

Const kCtrlrSwitch = "-ctrlr"
Const kSystemSwitch = "-system"
Const kDestSwitch = "-dest"
Const kDeleteTempFilesSwitch = "-cleanup"


'enum iomode for FileSystemObject
Const ForReading = 1
Const ForWriting = 2
Const ForAppending = 8

'enum format for FileSystemObject
Const OpenAsDefault = -2
Const OpenAsUnicode = -1
Const OpenAsASCII = 0

Private Function CopyUsingSystemFile(ByRef systemFile, ByRef sourceDir, byRef destDir)
    Dim returnMe
    returnMe = True
    
    ' Set up the three RegExps.
    Dim commentRE, blankRE, sectionRE, nameRE
    Dim theMatches, theSubs, aMatch, sectionName, currentSourcePath, fileName

    Set commentRE = New RegExp
    Set blankRE = New RegExp
    Set sectionRE = New RegExp
    Set nameRE = New RegExp
    commentRE.Pattern = kCommentPattern
    blankRE.Pattern = kBlankPattern
    sectionRE.Pattern = kSectionPattern
    nameRE.Pattern = kNamePattern
    
    ' Open the system file
    Dim fso, systemFileObject, sourceDirObject, currentSourceFile, textStream, thisLine
    Set fso = CreateObject("Scripting.FileSystemObject")
    Set systemFileObject = fso.GetFile(systemFile)
    Set textStream = systemFileObject.OpenAsTextStream(ForReading, OpenAsASCII)
    
    If Not fso.FolderExists(destDir) Then
        ' Bug: only works for one level of folder creation.
	Err.Clear
	On Error Resume Next
        Call fso.CreateFolder(destDir)
	If Err.Number <> 0 Then
	  returnMe = False
	  WScript.Echo(Err.Description & vbCrLf & destDir)
	End If
	On Error Goto 0
    End If
    
    'Do While the file isn't at EOF and there isn't an error:
    Do While (Not textStream.AtEndOfStream) And returnMe
   	thisLine = textStream.ReadLine

	If commentRE.Test(thisLine) Or blankRE.Test(thisLine) Then
	    ' do nothing
	ElseIf sectionRE.Test(thisLine) Then
	    Set theMatches = sectionRE.Execute(thisLine)
	    Set aMatch = theMatches(0)
	    Set theSubs = aMatch.SubMatches
	    sectionName = theSubs(0)
	    currentSourcePath = sourceDir & kSubDirChar & sectionName & kWriteToExtension
	    Set currentSourceFile = fso.GetFile(currentSourcePath)
	ElseIf nameRE.Test(thisLine) Then ' copy
	    Set theMatches = nameRE.Execute(thisLine)
	    Set aMatch = theMatches(0)
	    Set theSubs = aMatch.SubMatches
	    fileName = theSubs(0)
	    
	    fileName = fileName & kWriteToExtension
	    fileName = destDir & kSubDirChar & fileName
	    Call currentSourceFile.Copy(fileName, True)
	Else
	    returnMe = False
	    WScript.Echo(thisLine & vbCRLf & "Whoops!")
	End If
    Loop
    textStream.Close 

    CopyUsingSystemFile = returnMe
End Function

Private Function ReplaceInThisFile(ByRef sourceFile, ByRef destDir, ByRef replaceFrom, ByRef replaceWith)
    Dim returnMe
    returnMe = True

    Dim replaceMax
    replaceMax = UBound(replaceFrom, 1) - 1 

    Dim newFileName
    newFileName = Left(sourceFile.Name, Len(sourceFile.Name) - Len(kActOnExtension))
    newFileName = LCase(newFileName & kWriteToExtension)

    Dim sourceStream, destStream
    Set sourceStream = sourceFile.OpenAsTextStream(ForReading, OpenAsASCII)
    Set destStream = destDir.CreateTextFile(newFileName, True, False)

    Dim thisLine, i
    Do While Not sourceStream.AtEndOfStream
	thisLine = sourceStream.ReadLine
	For i = 0 to replaceMax
	    thisLine = Replace(thisLine, replaceFrom(i), replaceWith(i))
	Next
	destStream.WriteLine(thisLine)
    Loop

    destStream.Close
    sourceStream.Close
        
    ReplaceInThisFile = returnMe
End Function



Private Function ReplaceFromThisFolder(ByRef sourcePath, ByRef destPath, ByRef replaceFrom, ByRef replaceWith)
    Dim returnMe
    returnMe = True

    Dim fso, sourceDir, destDir
    Set fso = CreateObject("Scripting.FileSystemObject")
    Set sourceDir = fso.GetFolder(sourcePath)
    If Not (fso.FolderExists(destPath)) Then
    	fso.CreateFolder(destPath)
    End If
    Set destDir = fso.GetFolder(destPath)

    Dim sourceFiles, sourceFile, actLen
    Set sourceFiles = sourceDir.Files
    actLen = Len(kActOnExtension)
    For Each sourceFile in sourceFiles
	If Right(sourceFile.Name, actLen) = kActOnExtension Then
	    returnMe = returnMe And ReplaceInThisFile(sourceFile, destDir, replaceFrom, replaceWith)
	End If
    Next

    ReplaceFromThisFolder = returnMe
End Function



Private Function ReadVariables(ByRef filePath, ByRef replaceFrom, ByRef replaceWith)
    Dim returnMe
    returnMe = 0
    
    Dim fso, theFile, textStream
    Set fso = CreateObject("Scripting.FileSystemObject")
    Set theFile = fso.GetFile(filePath)
    Set textStream = theFile.OpenAsTextStream(ForReading, OpenAsASCII)

    Dim nextItem, maxDim, dimIncrement
    nextItem = 0
    maxDim = 10
    dimIncrement = 10
    ReDim replaceFrom(maxDim)
    ReDim replaceWith(maxDim)

    Dim thisLine

    Dim regEx, aMatch, theMatches

    Set regEx = New RegExp
    regEx.Pattern = kVariablePattern

    Do While Not textStream.AtEndOfStream
	thisLine = textStream.ReadLine

	Set theMatches = regEx.Execute(thisLine)
	If (theMatches.Count = 1) Then
	    Set aMatch = theMatches(0)
	    If aMatch.SubMatches.Count = 2 Then
		    If (nextItem = maxDim) Then
		    maxDim = maxDim + dimIncrement
		    ReDim Preserve replaceFrom(maxDim)
		    ReDim Preserve replaceWith(maxDim)
		End If
		replaceFrom(nextItem) = aMatch.SubMatches(0)
	        replaceWith(nextItem) = aMatch.SubMatches(1)
		nextItem = nextItem + 1
	    End If
	End If
    Loop
    textStream.Close

    ' Remember, nextItem is 1 more than the number of items in the array,
    ' which is the right number for ReDim.
    If (nextItem < (maxDim - 1)) Then
	maxDim = nextItem
	ReDim Preserve replaceFrom(maxDim)
	ReDim Preserve replaceWith(maxDim)
    End If

    returnMe = maxDim

    ReadVariables = returnMe
End Function
 

Private Function InterpretCommandLine(ByRef ctrlrName, ByRef systemName, ByRef sourceRoot, ByRef destRoot, ByRef tempRoot, ByRef cleanupTemp)
    sourceRoot = kDefaultSourceDir
    destRoot = kDefaultDestDir
    tempRoot = kDefaultTempDir
    ctrlrName = ""
    systemName = ""
    cleanupTemp = True

    Dim continueOn
    continueOn = True

    Dim i, args, thisSwitch, thisValue
    Set args = WScript.Arguments
    If (args.Count = 0) Then
	WScript.Echo(errArgumentsRequired)
	continueOn = False
    ElseIf (args.Count Mod 2 = 1) Then
	WScript.Echo(errIncorrectArgumentCount)
        continueOn = False
    Else
	For i = 0 to args.Count - 1 Step 2
	    thisSwitch = args(i)
	    thisValue = args(i+1)
	    If (thisSwitch = kCtrlrSwitch) Then
		ctrlrName = thisValue
	    ElseIf (thisSwitch = kSystemSwitch) Then
		systemName = thisValue
	    ElseIf (thisSwitch = kDestSwitch) Then
		destRoot = thisValue
	    End If
        Next
    End If

    If continueOn Then
        ' Fix: make sure directory paths don't end with \
	If Right(sourceRoot, 1) = "\" Then sourceRoot = Left(sourceRoot, Len(sourceRoot)-1)
        If Right(destRoot, 1) = "\" Then destRoot = Left(destRoot, Len(destRoot)-1)
	If Right(tempRoot, 1) = "\" Then tempRoot = Left(tempRoot, Len(tempRoot)-1)
	    
        If (ctrlrName = "") Then
	   WScript.Echo(errNoCtrlr)
	    continueOn = False
        ElseIf (systemName = "") Then
	    WScript.Echo(errNoSystem)
	    continueOn = False
	End If
    End If
    
    InterpretCommandLine = continueOn
End Function






Dim continueOn
Dim ctrlrName, systemName, sourceRoot, destRoot, tempRoot, cleanupTemp
continueOn = InterpretCommandLine(ctrlrName, systemName, sourceRoot, destRoot, tempRoot, cleanupTemp)

Dim replaceFrom(), replaceWith()
Dim replaceCount
If continueOn Then
    Dim systemPath, ctrlrPath, variablePath, destPath
    systemPath = sourceRoot & "\" & systemName & kDefinitionExtension
    ctrlrPath = sourceRoot & "\" & ctrlrName
    variablePath = ctrlrPath & "\" & ctrlrName & kDefinitionExtension
    destPath = destRoot & kSubDirChar & ctrlrName
    replaceCount = ReadVariables(variablePath, replaceFrom, replaceWith)
    If (replaceCount > 0) Then
        continueOn = ReplaceFromThisFolder(ctrlrPath, tempRoot, replaceFrom, replaceWith)
        If continueOn Then
	    continueOn = CopyUsingSystemFile(systemPath, tempRoot, destPath)
        End If
	If continueOn And cleanupTemp Then
	    Dim fso
	    Set fso = CreateObject("Scripting.FileSystemObject")
	    Call fso.DeleteFolder(tempRoot, true)
	End If
	If continueOn Then
	    WScript.Echo(msgComplete)
        End If
    Else
	WScript.Echo("Need to fix not having any variables to replace.")
    End If
End If



